<?php
/**
 *	WHMCS Payment Gateway - Coinify
 *
 *	@package     WHMCS
 *	@version     1.0.0
 *	@copyright   Copyright (c) ALC Media 2026
 *	@author      ALC Media <info@alc.media>
 *  @link        https://coinify.readme.io/docs/whmcs
 */

require '../../../init.php';

use WHMCS\Billing\Invoice;
use WHMCS\Module\GatewaySetting;
use WHMCS\Billing\Payment\Transaction;
use WHMCS\Billing\Payment\Transaction\History;
use WHMCS\Payment\Exception\InvalidModuleException;

global $CONFIG;

App::load_function('gateway');
App::load_function('invoice');
App::load_function('ccfunctions');

$return = [];
$extraLog = [];
$logTransaction = '';

$coinifyIPAddressList = [
    '34.240.144.187',
    '34.249.67.212',
    '34.251.158.97',
    '34.247.148.246',
    '34.247.148.57',
    '34.249.123.8',
];

try {
    if (!in_array($_SERVER['REMOTE_ADDR'], $coinifyIPAddressList)) {
        throw new Exception('Invalid Access Attempt');
    }

    $payload = file_get_contents('php://input');
    $params = getGatewayVariables('coinify');

    if (!$params['type']) {
        throw new InvalidModuleException('Module Not Activated');
    }

    $hash = hash_hmac('sha256', $payload, $params['sharedSecret']);
    $signature = $_SERVER['HTTP_X_COINIFY_WEBHOOK_SIGNATURE'] ?? '';
    
    if (!hash_equals($hash, $signature)) {
        throw new Exception('Invalid Signature');
    }

    $return = json_decode($payload, true);

    checkCbTransID($return['context']['id']);

    $invoice = null;
    $invoiceID = $return['context']['orderId'];

    if ($params['useInvoiceNumber']) {
        $invoice = Invoice::where('invoicenumber', $invoiceID)->first();

        if (!$invoice) {
            throw new Exception('Invoice Number Not Found');
        }

        $invoiceID = $invoice->id;
    } else {
        $invoice = Invoice::find($invoiceID);
    }

    if (!$invoice) {
        throw new Exception('Invoice Not Found');
    }

    $history = History::firstOrNew([
        'gateway' => $params['name'],
        'invoice_id' => $invoiceID,
        'transaction_id' => $return['context']['id']
    ]);

    $extraLog['history_id'] = $history->id;

    switch ($return['event']) {
        case 'payment-intent.completed':
            $history->remoteStatus = $return['context']['state'];
            $history->description = $return['context']['stateReason'];

            if ($return['context']['state'] !== 'completed') {
                $history->completed = false;
                $history->save();

                throw new Exception('Payment Not Completed');
            }

            $history->completed = true;
            $history->save();

            addInvoicePayment($invoiceID, $return['context']['id'], $return['context']['amount'], 0, $params['name']);

            $logTransaction = 'Invoice Paid';
            break;
        
        case 'payment-intent.failed':
            $history->remoteStatus = $return['context']['state'];
            $history->description = $return['context']['stateReason'];
            $history->completed = false;
            $history->save();

            $logTransaction = 'Payment Failed: ' . $return['context']['stateReason'];
            break;
    }
} catch (InvalidModuleException $e) {
    $return = ['error' => $e->getMessage()];
    $logTransaction = 'Module Not Active';
} catch (Exception $e) {
    $return = ['payload' => $return, 'error' => $e->getMessage()];

    if ($e->getMessage() === 'Invalid Signature') {
        $logTransaction = 'Invalid Access Attempt';
        http_response_code(403);
    } else {
        $logTransaction = 'Invalid Response';
        http_response_code(400);
    }
}

logTransaction('Coinify', $return, $logTransaction, $extraLog);

WHMCS\Terminus::getInstance()->doExit();